using System;
using System.Data;
using MethodBase = System.Reflection.MethodBase;
using DAILYCONTROL = gov.va.med.vbecs.Common.VbecsTables.DailyControl;

namespace gov.va.med.vbecs.BOL
{
	#region Header
		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/23/2003</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary>
		///Daily Control deals with the quality control applied to the reagent expected minimum ractions and which reagent types are in the racks
		///</summary>
	#endregion

	public class DailyControl : BOL.BaseBusinessObject, IRecordStatusCode
	{
		#region Variables

		private System.Guid _dailyControlGUID;
		private int _numberOfRacks;
		private bool _rackNamesIndicator;
		private System.DateTime _dailyAlertTime;
		private bool _commercialTemplateIndicator;
		private BOL.EnhancementMedia _primaryEnhancementMedia;
		private BOL.EnhancementMedia _secondaryEnhancementMedia;
		private System.Data.DataTable _dtDailyControls;
		private System.Data.DataTable _dtRackData;
		private System.Data.DataTable _dtRacks;
		private Common.RecordStatusCode _recordStatusCode;

		#endregion

		#region Constructors

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/3/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1586"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>A new daily control object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Default constructor for a BOL daily control
		/// </summary>
		public DailyControl()
		{
			//
			// TODO: Add constructor logic here
			//
			_primaryEnhancementMedia = new BOL.EnhancementMedia();
			_secondaryEnhancementMedia = new BOL.EnhancementMedia();
		}

		///<Developers>
		///	<Developer></Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7985"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7986"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Daily Control
		/// </summary>
		/// <param name="dailyControlGuid"></param>
		public DailyControl(System.Guid dailyControlGuid)
		{
			//
			// TODO: Add constructor logic here
			//
			System.Data.DataTable dt = DAL.DailyControl.GetDailyControlByGuid(dailyControlGuid);
			if(dt.Rows.Count != 0)
			{
				_primaryEnhancementMedia = new BOL.EnhancementMedia();
				_secondaryEnhancementMedia = new BOL.EnhancementMedia();
				this.LoadFromDataRow(dt.Rows[0]);
			}
			else
			{
				throw new BOL.BusinessObjectException("Daily Control GUID not found " + dailyControlGuid.ToString());
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2403"> 
		///		<ExpectedInput>DataRow containing DailyControl data</ExpectedInput>
		///		<ExpectedOutput>DailyControl object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2440"> 
		///		<ExpectedInput>Invalid DataRow</ExpectedInput>
		///		<ExpectedOutput>Invalid DailyControl object</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates DailyControl object and populates it with data from DataRow
		/// </summary>
		public DailyControl(DataRow dr)
		{
			_primaryEnhancementMedia = new BOL.EnhancementMedia();
			_secondaryEnhancementMedia = new BOL.EnhancementMedia();
			this.LoadFromDataRow(dr);
		}

		#endregion

		#region Properties

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1376"> 
		///		<ExpectedInput>A valid GUID</ExpectedInput>
		///		<ExpectedOutput>A valid GUID</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2093"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The GUID for the daily control
		/// </summary>
		public System.Guid DailyControlGuid
		{
			get 
			{
				return _dailyControlGUID;
			}
			set 
			{
				_dailyControlGUID = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1377"> 
		///		<ExpectedInput>An integer between 1 and 26</ExpectedInput>
		///		<ExpectedOutput>No broken rules</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2094"> 
		///		<ExpectedInput>An integer over 26</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The number of racks to QC daily
		/// Implemented BR_51.01
		/// </summary>
		public int NumberOfRacks
		{
			get 
			{
				return _numberOfRacks;
			}
			set 
			{
				RuleBroken(MethodBase.GetCurrentMethod().Name, (value >= 1 && value <= 26) == false);
				if(value >= 1 && value <= 26)
				{
					_numberOfRacks = value;
				}
				else
				{
					//BR_51.01
					string errorMessage = Common.StrRes.SysErrMsg.Common.RangeError("The number of racks", "1", "26").ResString;
					throw new BOL.BusinessObjectException(errorMessage);
				}
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1378"> 
		///		<ExpectedInput>True</ExpectedInput>
		///		<ExpectedOutput>No additional broken rules</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2097"> 
		///		<ExpectedInput>False</ExpectedInput>
		///		<ExpectedOutput>An additional broken rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The rack names indicator for the object
		/// </summary>
		public bool RackNamesIndicator
		{
			get 
			{
				return _rackNamesIndicator;
			}
			set 
			{
				//This just makes sure that the value is set because it 
					//is a required field
				RuleBroken(MethodBase.GetCurrentMethod().Name, false);
				_rackNamesIndicator = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6268"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6269"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// The status of the record
		/// </summary>
		public Common.RecordStatusCode RecordStatusCode
		{
			get
			{
				return _recordStatusCode;
			}
			set
			{
				_recordStatusCode = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1379"> 
		///		<ExpectedInput>Valid date and time</ExpectedInput>
		///		<ExpectedOutput>Valid date and time</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2099"> 
		///		<ExpectedInput>Invalid date and time</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///		
		/// <summary>
		/// Alert time 
		/// Implemented BR_51.02
		/// </summary>
		public System.DateTime DailyAlertTime
		{
			get 
			{
				return _dailyAlertTime;
			}
			set 
			{
				RuleBroken(MethodBase.GetCurrentMethod().Name,  value == DateTime.MinValue);
				if(value != DateTime.MinValue)
				{
					_dailyAlertTime = value;
				}
				else
				{
					//BR_51.02
					string errorMessage = Common.StrRes.SysErrMsg.Common.InvalidPropertyOrMethod(
											Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name, true),
											value).ResString;
					throw new BOL.BusinessObjectException(errorMessage);

				}
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1380"> 
		///		<ExpectedInput>Boolean</ExpectedInput>
		///		<ExpectedOutput>Boolean</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="2102"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Commercial Template Indicator for the object
		/// </summary>
		public bool CommercialTemplateIndicator
		{
			get 
			{
				return _commercialTemplateIndicator;
			}
			set 
			{
				_commercialTemplateIndicator = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1381"> 
		///		<ExpectedInput>Valid PrimaryEnhancementMedia</ExpectedInput>
		///		<ExpectedOutput>Valid PrimaryEnhancementMedia</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2103"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Primary Enhancement Media for the object
		/// </summary>
		public BOL.EnhancementMedia PrimaryEnhancementMedia
		{
			get 
			{
				return this._primaryEnhancementMedia;
			}
			set 
			{
				this._primaryEnhancementMedia = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1382"> 
		///		<ExpectedInput>Valid SecondaryEnhancementMedia</ExpectedInput>
		///		<ExpectedOutput>Valid SecondaryEnhancementMedia</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2104"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Secondary Enhancement Media for the object
		/// </summary>
		public BOL.EnhancementMedia SecondaryEnhancementMedia
		{
			get 
			{
				return this._secondaryEnhancementMedia;
			}
			set 
			{
				this._secondaryEnhancementMedia = value;
			}
		}

		///<Developers>
		///	<Developer>EDS EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7987"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7988"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///     BR_55.02
		/// </summary>
		/// 
		/// <returns>
		///     A bool value...
		/// </returns>
		public static bool PartiallyCompletedQC()
		{
			System.Data.DataTable dt = DAL.DailyControl.GetPartiallyCompletedQC();
			return (dt.Rows.Count > 0);
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6270"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6271"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetExistingDailyControlTemplate
		/// </summary>
		/// <returns>DailyControl object.</returns>
		public BOL.DailyControl GetExistingDailyControlTemplate()
		{
			_dtDailyControls = DAL.DailyControl.GetDailyControl();

			//There should be only one row
			if(_dtDailyControls.Rows.Count > 0)
			{
				return new BOL.DailyControl(_dtDailyControls.Rows[0]);
			}
			else
			{
				return null;
			}
		}

		#endregion

		private System.Data.DataTable GetAllRacks()
		{
			return _dtRacks = DAL.DailyControl.GetAllRacks();
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1399"> 
		///		<ExpectedInput>Valid rack data and reagent data tables</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2088"> 
		///		<ExpectedInput>Invalid rack data and reagent data tables</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Save daily controls
		/// </summary>
		/// <returns></returns>
		/// <param name="alReagentData"></param>
		/// <param name="commercialIndicator"></param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		public bool SaveAllDailyControlData(System.Collections.ArrayList alReagentData, bool commercialIndicator, Common.UpdateFunction lastUpdateFunctionId)
		{
			//Calling this with no division since I only want the structure of the table I 
			//have the data in the arraylist
			System.Data.DataTable dtReagentData = DAL.ReagentType.GetEmptyReagentTestParameterTableSchema(false);
			dtReagentData.Rows.Clear();
			foreach(System.Object obj in alReagentData)
			{
				System.Data.DataRow dtRow = ((BOL.ReagentType) obj).LoadDataRowFromThis(dtReagentData.NewRow());
				dtRow[Common.VbecsTables.ReagentTestParameter.CommercialIndicator] = commercialIndicator;
				dtRow[Common.VbecsTables.ReagentTestParameter.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
				dtReagentData.Rows.Add(dtRow);
			}
			
			System.Data.DataTable dtExistingDailyControl = DAL.DailyControl.GetDailyControl();
			System.Data.DataTable dtInsertDC = DAL.DailyControl.GetEmptyDailyControlTableSchema(false);
			System.Data.DataTable dtUpdateDC = DAL.DailyControl.GetEmptyDailyControlTableSchema(true);

			bool skipInsert = false;

			if (dtExistingDailyControl.Rows.Count > 0)
			{
				//Check to see if we're just updating what's out there
				if ((bool)dtExistingDailyControl.Rows[0][DAILYCONTROL.CommercialTemplateIndicator] == this.CommercialTemplateIndicator)
				{
					//TemplateIndicator is the same, so let's just update what's out there
					skipInsert = true;		//We can skip the insert, we're only here to do an update
					System.Data.DataRow drUpd = this.LoadDataRowFromThis(dtUpdateDC.NewRow());
					dtUpdateDC.Rows.Add(drUpd);
				}
				else
				{
					//TemplateIndicator is different, so we need to inactivate what's out there, we'll insert new outside this if/else
					System.Data.DataRow drInactivate = dtUpdateDC.NewRow();
					foreach(System.Data.DataColumn tableCol in dtExistingDailyControl.Columns)
					{
						if (drInactivate.Table.Columns.Contains(tableCol.ColumnName))
						{
							drInactivate[tableCol.ColumnName] = dtExistingDailyControl.Rows[0][tableCol.ColumnName];
						}
					}
					drInactivate[DAILYCONTROL.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
					dtUpdateDC.Rows.Add(drInactivate);
				}
			}

			//Insert what the user has entered -- only if we need to 
			if (!skipInsert)
			{
				System.Data.DataRow drIns = this.LoadDataRowFromThis(dtInsertDC.NewRow());
				dtInsertDC.Rows.Add(drIns);
			}
			
			return DAL.DailyControl.SaveAllDailyControlData(commercialIndicator, this._dtRackData, dtReagentData, lastUpdateFunctionId, dtInsertDC, dtUpdateDC);
		} 

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1400"> 
		///		<ExpectedInput>Valid DataTable</ExpectedInput>
		///		<ExpectedOutput>Valid DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2087"> 
		///		<ExpectedInput>Invalid DataTable</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Create and populate ck
		/// This method implements BR 51.13 and 51.14
		/// </summary>
		/// <returns></returns>
		public void PopulateRackData(System.Guid dailyControlGuid)
		{
			int nNumericRackName;
			char cAlphaRackName = 'A';
			int nCtr;
			System.Data.DataTable dtAllRacks = this.GetAllRacks();
			if (dailyControlGuid == Guid.Empty)
			{
				throw new BusinessObjectException("Daily control ID is empty");
			}

			_dtRackData = Rack.GetConfigurableRackQC(dailyControlGuid);
			//Compare the count of the current racks to the current request for racks
			//IF they had 10 and then they want 15
			if(_dtRackData.Rows.Count < this.NumberOfRacks)
			{
				nCtr = _dtRackData.Rows.Count;
				if(nCtr == 0)
				{
					nCtr = 1;
				}
				//
				nNumericRackName = nCtr;
				if(_dtRackData.Rows.Count > 0)
				{
					cAlphaRackName = System.Convert.ToChar(_dtRackData.Rows[nCtr - 1][Common.VbecsTables.Rack.RackName].ToString().Trim());
					nCtr++;
					nNumericRackName ++;
					cAlphaRackName++;
				}
				//
				for(int nNewCtr = nCtr; nNewCtr <= this.NumberOfRacks; nNewCtr++)
				{
					System.Data.DataRow drRack = _dtRackData.NewRow();
					drRack[DAILYCONTROL.DailyControlGuid] = this.DailyControlGuid;
					//
					if (this.RackNamesIndicator == true)
					{
						//alpha rack names
						drRack[Common.VbecsTables.Rack.RackName] = cAlphaRackName;
						cAlphaRackName++;
					}
					else
					{
						//numeric rack names
						drRack[Common.VbecsTables.Rack.RackName] = nNumericRackName;
						nNumericRackName++;
					}
					string rackName = drRack[Common.VbecsTables.Rack.RackName].ToString();
					System.Data.DataRow [] dataRows = dtAllRacks.Select("RackName = '" + rackName + "'");
					if(dataRows.Length == 1)
					{
						drRack[Common.VbecsTables.Rack.RackGuid] = dataRows[0][Common.VbecsTables.Rack.RackGuid];
						drRack[Common.VbecsTables.Rack.RowVersion] = dataRows[0][Common.VbecsTables.Rack.RowVersion];
//						for ( int idx = 0; idx < dtAllRacks.Rows.Count; idx++ )
//						{
//							Guid g1 = (Guid)dtAllRacks.Rows[idx][Common.VbecsTables.Rack.RackGuid];
//							Guid g2 = (Guid)drRack[Common.VbecsTables.Rack.RackGuid];
//							if(g1.Equals(g2))
//							{
//								drRack[Common.VbecsTables.Rack.RowVersion] = dtAllRacks.Rows[idx][Common.VbecsTables.Rack.RowVersion];
//							}
//						}
						
					}
					else
					{
						drRack[Common.VbecsTables.Rack.RackGuid] = System.Guid.NewGuid();
					}
					drRack[Common.VbecsTables.Rack.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
					drRack[Common.VbecsTables.Rack.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					_dtRackData.Rows.Add(drRack);
				}
			}
			//If they have decremented the number of racks then we must inactivate some
			//They had 15 now they have 10 we have to inactivate the last 5
			else if(_dtRackData.Rows.Count > this.NumberOfRacks)
			{
				//They decremented by 5
				int nDecrement = _dtRackData.Rows.Count - this.NumberOfRacks;
				int nIndx = _dtRackData.Rows.Count - nDecrement;
				int nCount = 0;
				nCount = _dtRackData.Rows.Count;
				for(int nNewIndx = nIndx; nNewIndx < nCount; nNewIndx++)
				{
					if(this.RackNamesIndicator)
					{
						_dtRackData.Rows[nNewIndx][Common.VbecsTables.Rack.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
					}
					//If the racks are numbered there is no way to sort and garuntee the order so we need to select by rack name equalling the number
					else
					{
						int nTest = nNewIndx + 1;
						DataRow[] dataRows = _dtRackData.Select("RackName = '" + nTest + "'");
						dataRows[dataRows.GetUpperBound(0)][Common.VbecsTables.Rack.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
					}
				}
			}
		}
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/5/2003</CreationDate>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Load data row into an DailyControl object
		/// </summary>
		/// <param name="dtRow"></param>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			this.DailyControlGuid = (System.Guid) dtRow[DAILYCONTROL.DailyControlGuid];
			this.NumberOfRacks = System.Convert.ToInt32(dtRow[DAILYCONTROL.NumberOfRacks].ToString());
			this.RackNamesIndicator = (bool) dtRow[DAILYCONTROL.RackNamesIndicator];
			this.DailyAlertTime = (System.DateTime) dtRow[DAILYCONTROL.DailyAlertTime];
			this.CommercialTemplateIndicator = (bool) dtRow[DAILYCONTROL.CommercialTemplateIndicator];
			this.PrimaryEnhancementMedia.EnhancementMediaId = (int) dtRow[DAILYCONTROL.PrimaryEnhancementMediaId];
			this.SecondaryEnhancementMedia.EnhancementMediaId = dtRow.IsNull(DAILYCONTROL.SecondaryEnhancementMediaId) ? Int32.MinValue : (int) dtRow[DAILYCONTROL.SecondaryEnhancementMediaId];
			this.Comments = dtRow.IsNull(DAILYCONTROL.Comments) ? "" : dtRow[DAILYCONTROL.Comments].ToString();
			this.RecordStatusCode = Common.Utility.GetRecordStatusCodeFromString(dtRow[DAILYCONTROL.RecordStatusCode].ToString());
			if(dtRow.Table.Columns.Contains(DAILYCONTROL.RowVersion))
			{
				this.RowVersion = (byte []) dtRow[DAILYCONTROL.RowVersion];
			}
			this.IsNew = false;
		}
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/5/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1597"> 
		///		<ExpectedInput>A valid data row</ExpectedInput>
		///		<ExpectedOutput>A populated data row from the existing daily control object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2085"> 
		///		<ExpectedInput>An invalid data row</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Load the DailyControl object into the data row
		/// </summary>
		/// <param name="dtRow"></param>
		/// <returns></returns>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			if (dtRow.Table.Columns.Contains(DAILYCONTROL.DailyControlGuid))
			{
				dtRow[DAILYCONTROL.DailyControlGuid] = this.DailyControlGuid;
			}

			if (dtRow.Table.Columns.Contains(DAILYCONTROL.NumberOfRacks))
			{
				dtRow[DAILYCONTROL.NumberOfRacks] = this.NumberOfRacks;
			}
			
			if (dtRow.Table.Columns.Contains(DAILYCONTROL.RackNamesIndicator))
			{
				dtRow[DAILYCONTROL.RackNamesIndicator] = this.RackNamesIndicator;
			}

			if (dtRow.Table.Columns.Contains(DAILYCONTROL.DailyAlertTime))
			{
				dtRow[DAILYCONTROL.DailyAlertTime] = this.DailyAlertTime;
			}

			if (dtRow.Table.Columns.Contains(DAILYCONTROL.CommercialTemplateIndicator))
			{
				dtRow[DAILYCONTROL.CommercialTemplateIndicator] = this.CommercialTemplateIndicator;
			}
			
			if (dtRow.Table.Columns.Contains(DAILYCONTROL.PrimaryEnhancementMediaId))
			{		
				dtRow[DAILYCONTROL.PrimaryEnhancementMediaId] = this.PrimaryEnhancementMedia.EnhancementMediaId;
			}
			
			if (dtRow.Table.Columns.Contains(DAILYCONTROL.SecondaryEnhancementMediaId))
			{
				if (this.SecondaryEnhancementMedia.EnhancementMediaId > 0)
				{
					dtRow[DAILYCONTROL.SecondaryEnhancementMediaId] = this.SecondaryEnhancementMedia.EnhancementMediaId;
				}
			}
			
			if (dtRow.Table.Columns.Contains(DAILYCONTROL.DivisionCode))
			{		
				dtRow[DAILYCONTROL.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			}
			

			if (dtRow.Table.Columns.Contains(DAILYCONTROL.Comments))
			{
				if (this.Comments != null && this.Comments != string.Empty)
				{
					dtRow[DAILYCONTROL.Comments] = this.Comments;
				}
			}

			if (dtRow.Table.Columns.Contains(DAILYCONTROL.RecordStatusCode))
			{	
				dtRow[DAILYCONTROL.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(this.RecordStatusCode);
			}
			
			if(dtRow.Table.Columns.Contains(DAILYCONTROL.RowVersion))
			{
				dtRow.Table.Columns[DAILYCONTROL.RowVersion].ReadOnly = false;
				if (this.RowVersion == null)
				{
					dtRow[DAILYCONTROL.RowVersion] = System.DBNull.Value;
				}
				else
				{
					dtRow[DAILYCONTROL.RowVersion] = this.RowVersion;
				}
			}

			return dtRow;
		}



	}
}



